from machine import Pin, PWM
import time

# ---- 調整パラメータ ----
LED_PWM_FREQ = 1000          # 1kHz（ちらつき防止）
BRIGHT_PCT   = 0.03          # 明るさ(0.0〜1.0)。眩しい場合は 0.01 など更に下げる
BRIGHT       = int(65535 * BRIGHT_PCT)

# ---- 配線定義 ----
SW1_PINS = [0, 1, 2, 3]                               # 4bit  LSB=GP0
SW2_PINS = [4, 5, 6, 7]                               # 4bit  LSB=GP4
SW3_PINS = [8, 9, 10, 11, 12, 13, 14, 15]             # 8bit  LSB=GP8
LED_RED  = PWM(Pin(16))                                # 誤り
LED_BLUE = PWM(Pin(17))                                # 正解

for led in (LED_RED, LED_BLUE):
    led.freq(LED_PWM_FREQ)
    led.duty_u16(0)  # 初期は消灯

# 入力（内部プルアップ）
def make_inputs(gpios):
    return [Pin(n, Pin.IN, Pin.PULL_UP) for n in gpios]

SW1 = make_inputs(SW1_PINS)
SW2 = make_inputs(SW2_PINS)
SW3 = make_inputs(SW3_PINS)

# ---- ユーティリティ ----
def read_bits(pins):
    """PULL_UP前提。物理SWはONで0V→数値上はON=1になるよう反転。LSBがpins[0]。"""
    val = 0
    for i, p in enumerate(pins):
        bit = 0 if p.value() == 0 else 1   # 物理値（ON=0, OFF=1）
        bit ^= 1                            # 数値として ON=1, OFF=0 にする
        val |= (bit << i)
    return val

def debounced_read(ms=25):
    a1 = (read_bits(SW1), read_bits(SW2), read_bits(SW3))
    time.sleep_ms(ms)
    a2 = (read_bits(SW1), read_bits(SW2), read_bits(SW3))
    return a2 if a1 == a2 else None

def set_leds(ok=False, err=False):
    """PWMで眩しさ軽減。OK=青、NG=赤。両方Falseで両消灯。"""
    LED_BLUE.duty_u16(BRIGHT if ok  else 0)
    LED_RED.duty_u16 (BRIGHT if err else 0)

def choose_op():
    print("演算を選択してください: +  -  *  //（整数割り算）")
    while True:
        s = input("> ").strip()
        if s in {"+", "-", "*", "//"}:
            return s
        print("無効です。+ / - / * / // から選んでください。")

def compute(a, b, op):
    if op == "+":  return (a + b) & 0xFF, None
    if op == "-":  return (a - b) & 0xFF, None
    if op == "*":  return (a * b) & 0xFF, None
    if op == "//":
        if b == 0:
            return None, "DIV0"
        return (a // b) & 0xFF, None

# ---- メイン ----
op = choose_op()
print(f"選択: {op}")
print("=== SW 学習モード ===")
print("SW1:GP0-3 / SW2:GP4-7 / SW3:GP8-15（8bit）")
print("正解=青LED(GP17) / 不正解=赤LED(GP16) / 0除算=赤高速点滅")

last = (-1, -1, -1)
blink_t = 0
blink_on = False

try:
    while True:
        vals = debounced_read(25)
        if vals:
            a, b, c = vals
            if vals != last:
                res, err = compute(a, b, op)
                if err == "DIV0":
                    # 0除算：青消灯、赤を高速点滅は下面のタイマー処理で実施
                    set_leds(False, False)
                    print(f"A={a:02d}  B={b:02d}  RES=---  SW3={c:03d}  [ERROR: divide by zero]")
                else:
                    ok = (res == c)
                    set_leds(ok, not ok)
                    print(f"A={a:02d}  B={b:02d}  ({op}) -> {res:03d}  SW3={c:03d}  {'OK' if ok else 'NG'}")
                last = vals

        # 0除算中の赤点滅（約8Hz）
        if last != (-1, -1, -1):
            a, b, _ = last
            _, err = compute(a, b, op)
            if err == "DIV0":
                now = time.ticks_ms()
                if now - blink_t > 120:
                    blink_t = now
                    blink_on = not blink_on
                    LED_BLUE.duty_u16(0)
                    LED_RED.duty_u16(BRIGHT if blink_on else 0)

        time.sleep_ms(15)
finally:
    # 安全のため終了時は消灯
    set_leds(False, False)
