# 距離に応じてLED本数を変える（MicroPython / Raspberry Pi Pico）
# 接続:
#   HC-SR04: TRIG=GP19, ECHO=GP18
#   LEDs: D3=GP21, D4=GP22, D5=GP26, D6=GP27 （各150Ω直列）
#
# ルール:
#  ≤10cm: D3
#  10–20cm: D3,D4
#  20–30cm: D3,D4,D5
#  ≥30cm: D3,D4,D5,D6

from machine import Pin, PWM, time_pulse_us
import time

# -------- 設定（好みに応じて調整） --------
LED_PINS = [21, 22, 26, 27]   # D3, D4, D5, D6
PWM_FREQ = 1000               # ちらつきにくい周波数
DUTY_ON  = 10000              # 明るさ（0〜65535）。暗めに設定
DUTY_OFF = 0
TRIG_PIN = 19
ECHO_PIN = 18
MEASURE_TIMEOUT_US = 30000    # 30ms ≒ 約5m
MEASURE_INTERVAL_S = 0.2

# -------- 初期化 --------
led_pwms = []
for p in LED_PINS:
    pwm = PWM(Pin(p, Pin.OUT))
    pwm.freq(PWM_FREQ)
    pwm.duty_u16(DUTY_OFF)
    led_pwms.append(pwm)

TRIG = Pin(TRIG_PIN, Pin.OUT, value=0)
ECHO = Pin(ECHO_PIN, Pin.IN)

# -------- 距離計測 --------
def measure_distance_cm():
    # 安定化
    TRIG.value(0)
    time.sleep_us(2)
    # 10usのトリガーパルス
    TRIG.value(1)
    time.sleep_us(10)
    TRIG.value(0)

    # EchoのHigh時間を測る
    dur = time_pulse_us(ECHO, 1, MEASURE_TIMEOUT_US)
    if dur <= 0:
        return None  # タイムアウトやエラー

    # 音速 343.2 m/s → 0.03432 cm/µs、往復なので /2
    dist_cm = dur * 0.01716
    return dist_cm

# -------- LED制御 --------
def set_led_count(n_on):
    # 先頭n個を点灯、残り消灯
    for i, pwm in enumerate(led_pwms):
        pwm.duty_u16(DUTY_ON if i < n_on else DUTY_OFF)

def update_leds_by_distance(d):
    # 仕様の境界をそのまま実装（≤, < の扱いに注意）
    if d <= 10:
        set_led_count(1)  # D3
    elif d <= 20:
        set_led_count(2)  # D3, D4
    elif d <= 30:
        set_led_count(3)  # D3, D4, D5
    else:
        set_led_count(4)  # D3, D4, D5, D6

# -------- メインループ --------
try:
    while True:
        dist = measure_distance_cm()
        if dist is None:
            # 測定失敗時は全部消灯（必要なら直前状態維持に変更可）
            set_led_count(0)
            print("Distance: -- cm (timeout)")
        else:
            update_leds_by_distance(dist)
            print("Distance: %.1f cm" % dist)
        time.sleep(MEASURE_INTERVAL_S)

except KeyboardInterrupt:
    # 終了時は消灯
    set_led_count(0)
    for pwm in led_pwms:
        pwm.deinit()
